// ==UserScript==
// @name         Close.com Lead Source Color & Display
// @namespace    http://tampermonkey.net/
// @version      3.8
// @description  Color-code leads by source and display lead source prominently
// @match        https://app.close.com/lead/*
// @grant        none
// @run-at       document-start
// ==/UserScript==

(function() {
    'use strict';

    console.log('[Lead Source Script] Script loaded');

    const leadSourceColors = {
        'Affordables': '#E0F2F1',           // Light teal
        'Burgenland AT': '#FFF9C4',         // Light yellow
        'C-Level Liste': '#E8F5E9',         // Light green
        'Digitalisierungshilfe': '#E3F2FD', // Light blue
        'Handmade': '#EFEBE9',              // Light brown
        'Meta Ads Liste': '#FFF3E0',        // Light orange
        'Other': '#F5F5F5',                 // Light gray
        'Provipanda': '#FCE4EC',            // Light pink
        'Startups Liste': '#F3E5F5',        // Light purple
        'Visual Conversion': '#EDE7F6',     // Light indigo
        'Default': '#FAFAFA'                // Off-white
    };

    let currentLeadSource = null;
    let currentLeadId = null;
    let applyTimeout = null;
    let isApplying = false;
    let leadSourceCache = {}; // Cache for lead sources
    let pendingLeadFetches = {}; // Track in-progress fetches

    // Extract lead ID from URL
    function getLeadIdFromUrl() {
        const match = location.href.match(/\/lead\/(lead_[^\/]+)/);
        return match ? match[1] : null;
    }

    // Fetch lead source directly from API
    async function fetchLeadSource(leadId) {
        if (pendingLeadFetches[leadId]) {
            console.log('[Lead Source Script] Already fetching:', leadId);
            return pendingLeadFetches[leadId];
        }

        console.log('[Lead Source Script] Fetching lead source for:', leadId);

        const fetchPromise = originalFetch(`/api/v1/lead/${leadId}/`, {
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            const leadSource = data.custom && data.custom['Lead Source'];
            if (leadSource) {
                leadSourceCache[leadId] = leadSource;
                console.log('[Lead Source Script] ✓ Fetched and cached Lead Source for', leadId, ':', leadSource);

                // Apply to call bar immediately
                applyToCallBar();

                return leadSource;
            }
            return null;
        })
        .catch(e => {
            console.log('[Lead Source Script] Error fetching lead:', e);
            return null;
        })
        .finally(() => {
            delete pendingLeadFetches[leadId];
        });

        pendingLeadFetches[leadId] = fetchPromise;
        return fetchPromise;
    }

    // Hook fetch to intercept lead API calls
    const originalFetch = window.fetch;
    window.fetch = function(...args) {
        const request = args[0];
        let url;

        if (typeof request === 'string') {
            url = request;
        } else if (request instanceof URL) {
            url = request.href;
        } else if (request instanceof Request) {
            url = request.url;
        } else if (request && request.url) {
            url = request.url;
        }

        if (url && url.includes('/api/v1/lead/lead_')) {
            // Extract lead ID from API URL
            const leadIdMatch = url.match(/\/lead\/(lead_[^\/\?]+)/);
            const apiLeadId = leadIdMatch ? leadIdMatch[1] : null;

            console.log('[Lead Source Script] ✓ Lead API detected for:', apiLeadId);

            return originalFetch.apply(this, args).then(response => {
                const clonedResponse = response.clone();
                clonedResponse.json().then(data => {
                    const leadSource = data.custom && data.custom['Lead Source'];

                    if (leadSource && apiLeadId) {
                        // Cache the lead source
                        leadSourceCache[apiLeadId] = leadSource;
                        console.log('[Lead Source Script] ✓ Cached Lead Source for', apiLeadId, ':', leadSource);

                        // Only update current if this is for the current URL's lead
                        const urlLeadId = getLeadIdFromUrl();
                        if (apiLeadId === urlLeadId) {
                            currentLeadSource = leadSource;
                            currentLeadId = apiLeadId;
                            console.log('[Lead Source Script] ✓ Lead Source for', apiLeadId, ':', leadSource);
                            scheduleApply();
                        }

                        // Try to apply to call bar if visible
                        applyToCallBar();
                    }
                }).catch(e => console.log('[Lead Source Script] Error:', e));

                return response;
            });
        }

        return originalFetch.apply(this, args);
    };

    function scheduleApply() {
        // Clear any pending application
        if (applyTimeout) {
            clearTimeout(applyTimeout);
        }

        // Apply immediately
        applyCustomizations();

        // Then retry a few times in case DOM isn't ready
        applyTimeout = setTimeout(() => applyCustomizations(), 100);
        setTimeout(() => applyCustomizations(), 300);
        setTimeout(() => applyCustomizations(), 600);
        setTimeout(() => applyCustomizations(), 1000);
    }

    function applyBackgroundColor(leadSource) {
        const color = leadSourceColors[leadSource] || leadSourceColors['Default'];
        const appContent = document.querySelector('.app-content');

        if (appContent) {
            if (appContent.style.backgroundColor !== color) {
                appContent.style.backgroundColor = color;
                appContent.style.transition = 'background-color 0.3s ease';
                console.log('[Lead Source Script] ✓ Background applied:', color);
            }
            return true;
        }
        return false;
    }

    function addLeadSourceBadge(leadSource) {
        if (!leadSource) return false;

        // Check if badge already exists and is correct
        const existingBadge = document.getElementById('custom-lead-source-badge');
        if (existingBadge && existingBadge.textContent === leadSource) {
            return true;
        }

        if (existingBadge) {
            existingBadge.remove();
        }

        // Find the flex container that holds the h1
        const nameRowContainer = document.querySelector('.LeadHeader_mainContent_542 .structure_utils_root_7ff.structure_utils_gap-8_249.structure_utils_align-center_c60.structure_utils_direction-row_38b h1');

        if (nameRowContainer && nameRowContainer.parentElement) {
            const color = leadSourceColors[leadSource] || leadSourceColors['Default'];

            const badge = document.createElement('span');
            badge.id = 'custom-lead-source-badge';
            badge.textContent = leadSource;
            badge.style.fontSize = '14px';
            badge.style.fontWeight = '600';
            badge.style.color = '#333';
            badge.style.backgroundColor = color;
            badge.style.padding = '4px 10px';
            badge.style.marginLeft = '12px';
            badge.style.borderRadius = '4px';
            badge.style.display = 'inline-block';
            badge.style.verticalAlign = 'middle';
            badge.style.border = '1px solid rgba(0, 0, 0, 0.1)';

            // Insert after the h1 as a sibling
            nameRowContainer.parentElement.appendChild(badge);
            console.log('[Lead Source Script] ✓ Badge added');
            return true;
        }
        return false;
    }

    function applyToCallBar() {
        // Find the call bar
        const callBar = document.querySelector('.CallBar.CallBar--active');
        if (!callBar) {
            return false;
        }

        // Find the lead link in the call bar to extract lead ID
        const leadLink = callBar.querySelector('.components_Contact_leadName_68d');
        if (!leadLink) {
            return false;
        }

        const leadIdMatch = leadLink.href.match(/\/lead\/(lead_[^\/]+)/);
        if (!leadIdMatch) {
            return false;
        }

        const leadId = leadIdMatch[1];
        let leadSource = leadSourceCache[leadId];

        if (!leadSource) {
            console.log('[Lead Source Script] No cached lead source for call bar lead:', leadId, '- fetching...');
            // Fetch the lead source directly
            fetchLeadSource(leadId);
            return false;
        }

        // Check if badge already exists
        const existingBadge = callBar.querySelector('#custom-call-bar-lead-source-badge');
        if (existingBadge && existingBadge.textContent === leadSource) {
            return true;
        }

        if (existingBadge) {
            existingBadge.remove();
        }

        // Find the lead row container
        const leadRow = callBar.querySelector('.components_Contact_leadRow_d10');
        if (!leadRow) {
            return false;
        }

        const color = leadSourceColors[leadSource] || leadSourceColors['Default'];

        const badge = document.createElement('span');
        badge.id = 'custom-call-bar-lead-source-badge';
        badge.textContent = leadSource;
        badge.className = 'Badge_outer_472 Badge_badge_7e7 Badge_small_877 Badge_root_140';
        badge.style.backgroundColor = color;
        badge.style.color = '#333';
        badge.style.fontSize = '12px';
        badge.style.fontWeight = '600';
        badge.style.padding = '2px 8px';
        badge.style.marginLeft = '8px';
        badge.style.borderRadius = '4px';
        badge.style.display = 'inline-block';
        badge.style.border = '1px solid rgba(0, 0, 0, 0.1)';

        leadRow.appendChild(badge);
        console.log('[Lead Source Script] ✓ Call bar badge added');
        return true;
    }

    function applyCustomizations() {
        // Prevent concurrent applications
        if (isApplying) return;
        isApplying = true;

        if (!currentLeadSource) {
            isApplying = false;
            return;
        }

        // Verify we're still on the same lead
        const urlLeadId = getLeadIdFromUrl();
        if (urlLeadId !== currentLeadId) {
            console.log('[Lead Source Script] Lead ID mismatch, skipping application');
            isApplying = false;
            return;
        }

        const bgApplied = applyBackgroundColor(currentLeadSource);
        const badgeApplied = addLeadSourceBadge(currentLeadSource);

        if (bgApplied && badgeApplied) {
            console.log('[Lead Source Script] ✓ Customizations applied');
        }

        isApplying = false;
    }

    // Watch for DOM changes to reapply badge if needed
    const observer = new MutationObserver((mutations) => {
        if (currentLeadSource && !document.getElementById('custom-lead-source-badge')) {
            const nameContainer = document.querySelector('.LeadHeader_mainContent_542 .structure_utils_root_7ff.structure_utils_gap-8_249.structure_utils_align-center_c60.structure_utils_direction-row_38b h1');
            if (nameContainer) {
                console.log('[Lead Source Script] Badge missing, reapplying...');
                applyCustomizations();
            }
        }

        // Check for call bar and apply badge if needed
        applyToCallBar();
    });

    // Start observing once DOM is ready
    if (document.body) {
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    } else {
        document.addEventListener('DOMContentLoaded', () => {
            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
        });
    }

    // Monitor URL changes
    let lastUrl = location.href;
    setInterval(() => {
        if (location.href !== lastUrl) {
            const newLeadId = getLeadIdFromUrl();
            console.log('[Lead Source Script] URL changed, new lead:', newLeadId);

            lastUrl = location.href;

            // Clear old state
            if (applyTimeout) clearTimeout(applyTimeout);

            // Remove old customizations
            const badge = document.getElementById('custom-lead-source-badge');
            if (badge) badge.remove();

            const appContent = document.querySelector('.app-content');
            if (appContent) appContent.style.backgroundColor = '';

            // Reset state - API call will set new values
            currentLeadSource = null;
            currentLeadId = null;
        }

        // Always try to apply to call bar if it's visible
        applyToCallBar();
    }, 200);

    console.log('[Lead Source Script] Ready');
})();
