(function() {
    'use strict';

    console.log('[Lead Source Extension] Extension loaded');

    // Project field ID and colors
    const PROJECT_FIELD_ID = 'cf_nlOUKlDeqwaZ4kXuSwNyWMHyRtObYonVyMNJvRhq5K4';
    const PROJECT_FIELD_NAME = 'Project';

    const projectColors = {
        'Digitalisierungshilfe': '#E3F2FD', // Light blue
        'Rebound Dating': '#FCE4EC',        // Light pink
        'Rebound Sales': '#C8E6C9',         // Light green
        'Visual Conversion': '#EDE7F6',     // Light purple
        'Default': '#FAFAFA'                // Off-white
    };

    // Phone number configuration based on project
    const phoneNumberMapping = {
        'Digitalisierungshilfe': '+436703061778',
        'Rebound Dating': '+4921295789999',
        'Rebound Sales': '+4921295789999',
        'Visual Conversion': '+4921295789999',
        'default': '+4921295789999'
    };

    // Normalize phone number by removing spaces, dashes, and parentheses
    function normalizePhone(phone) {
        if (!phone) return '';
        return phone.replace(/[\s\-\(\)]/g, '');
    }

    // Find available number that matches (handles format differences)
    function findMatchingNumber(requiredNumber) {
        const normalizedRequired = normalizePhone(requiredNumber);
        for (const [phone, info] of Object.entries(availableNumbers)) {
            if (normalizePhone(phone) === normalizedRequired) {
                return phone;
            }
        }
        return null;
    }

    // Phone number state
    let currentOutgoingNumber = null;
    let availableNumbers = {};
    let membershipInfo = null;
    let activeMembershipId = null;
    let pendingCallRequests = [];

    let currentProject = null;
    let currentLeadId = null;
    let currentLastCaller = null;
    let applyTimeout = null;
    let isApplying = false;
    let projectCache = {};
    let lastCallerCache = {};
    let pendingLeadFetches = {};

    // Caller tracking
    let currentCallerName = 'Mario'; // Default
    const LAST_CALLER_FIELD_ID = 'cf_T636zSWb5Mspu5PKkaciEJep9HUYrgzO9w8U4aZexc0';
    const LAST_CALLER_FIELD_NAME = 'Last Caller'; // Display name used in API responses

    // Helper function to get required phone number for project
    function getRequiredPhoneNumber(project) {
        const required = phoneNumberMapping[project] || phoneNumberMapping['default'];
        console.log('[Lead Source Extension] Required number for', project, ':', required);
        return required;
    }

    // Status display functions
    function updateStatusDisplay(text, isTemporary = false) {
        const statusElement = document.getElementById('custom-phone-status');
        if (statusElement) {
            statusElement.textContent = text;
        } else {
            createStatusDisplay(text);
        }

        // If temporary, revert to number display after delay
        if (isTemporary) {
            setTimeout(() => {
                if (currentOutgoingNumber && availableNumbers[currentOutgoingNumber]) {
                    updateStatusDisplay(availableNumbers[currentOutgoingNumber].formatted);
                }
            }, 2000);
        }
    }

    function createStatusDisplay(text) {
        // Find the TopNav search box container
        const searchBoxContainer = document.querySelector('.TopNav_searchBoxContainer_9a8');
        if (!searchBoxContainer) {
            return;
        }

        // Make search box container position relative
        searchBoxContainer.style.position = 'relative';

        // Check if status element already exists
        let statusElement = document.getElementById('custom-phone-status');
        if (statusElement) {
            statusElement.textContent = text;
            return;
        }

        // Create new status element with absolute positioning
        statusElement = document.createElement('div');
        statusElement.id = 'custom-phone-status';
        statusElement.textContent = text;
        statusElement.style.position = 'absolute';
        statusElement.style.top = '10px';
        statusElement.style.right = '10px';
        statusElement.style.fontSize = '13px';
        statusElement.style.fontWeight = '600';
        statusElement.style.color = '#555';
        statusElement.style.margin = '0';
        statusElement.style.padding = '0';
        statusElement.style.whiteSpace = 'nowrap';
        statusElement.style.pointerEvents = 'none';

        // Append to the search box container
        searchBoxContainer.appendChild(statusElement);

        console.log('[Lead Source Extension] ✓ Status display created');
    }

    // Function to fetch telephony settings manually
    async function fetchTelephonySettings() {
        console.log('[Lead Source Extension] Fetching telephony settings...');
        try {
            const response = await originalFetch('https://app.close.com/api/v1/graphql/?operationName=TelephonySettingsQuery', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    ...getAuthHeaders()
                },
                body: JSON.stringify({
                    operationName: 'TelephonySettingsQuery',
                    query: 'query TelephonySettingsQuery { viewer { user { id memberships { id defaultCallerId { id } groupNumbers { id number { phone formatted } } personalNumbers { id number { phone formatted } } organization { id isActive } } } } }'
                })
            });

            const data = await response.json();
            processTelephonySettings(data);
        } catch (e) {
            console.log('[Lead Source Extension] Error fetching telephony settings:', e);
        }
    }

    // Process telephony settings response
    function processTelephonySettings(data) {
        if (!data || !data.data || !data.data.viewer || !data.data.viewer.user) {
            return;
        }

        const memberships = data.data.viewer.user.memberships;
        const activeMembership = memberships.find(m => m.organization.isActive);

        if (!activeMembership) {
            console.log('[Lead Source Extension] No active membership found');
            return;
        }

        activeMembershipId = activeMembership.id;
        membershipInfo = {
            organizationId: activeMembership.organization.id,
            defaultCallerId: activeMembership.defaultCallerId?.id
        };

        // Build number map
        availableNumbers = {};

        // Add group numbers
        activeMembership.groupNumbers.forEach(num => {
            const phone = num.number.phone;
            availableNumbers[phone] = {
                id: num.id,
                formatted: num.number.formatted,
                type: 'group'
            };
        });

        // Add personal numbers
        activeMembership.personalNumbers.forEach(num => {
            const phone = num.number.phone;
            availableNumbers[phone] = {
                id: num.id,
                formatted: num.number.formatted,
                type: 'personal'
            };
        });

        // Determine current outgoing number
        if (membershipInfo.defaultCallerId) {
            const currentNum = Object.entries(availableNumbers).find(([phone, info]) => info.id === membershipInfo.defaultCallerId);
            if (currentNum) {
                currentOutgoingNumber = currentNum[0];
                console.log('[Lead Source Extension] ✓ Current outgoing number:', availableNumbers[currentOutgoingNumber].formatted);
                // Update status display
                updateStatusDisplay(availableNumbers[currentOutgoingNumber].formatted);
            }
        }

        console.log('[Lead Source Extension] ✓ Available numbers:', Object.values(availableNumbers).map(n => n.formatted));
    }

    // Function to update outgoing number
    async function updateOutgoingNumber(requiredNumber) {
        // Find the matching number in availableNumbers (handles format differences)
        const matchingPhone = findMatchingNumber(requiredNumber);
        const numberInfo = matchingPhone ? availableNumbers[matchingPhone] : null;

        if (!numberInfo) {
            console.log('[Lead Source Extension] ✗ Required number not available:', requiredNumber, '(normalized:', normalizePhone(requiredNumber), ')');
            console.log('[Lead Source Extension] Available numbers:', Object.keys(availableNumbers).map(p => normalizePhone(p)));
            return false;
        }

        if (!activeMembershipId || !membershipInfo) {
            console.log('[Lead Source Extension] ✗ No membership info available');
            return false;
        }

        console.log('[Lead Source Extension] Updating outgoing number to:', numberInfo.formatted);
        updateStatusDisplay('Updating number...');

        try {
            const response = await originalFetch(`https://app.close.com/api/v1/membership/${activeMembershipId}/`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    ...getAuthHeaders()
                },
                body: JSON.stringify({
                    default_caller_id: numberInfo.id,
                    organization_id: membershipInfo.organizationId
                })
            });

            if (response.ok) {
                currentOutgoingNumber = matchingPhone;  // Use the actual key from availableNumbers
                membershipInfo.defaultCallerId = numberInfo.id;
                console.log('[Lead Source Extension] ✓ Outgoing number updated to:', numberInfo.formatted);
                updateStatusDisplay(numberInfo.formatted);
                return true;
            } else {
                console.log('[Lead Source Extension] ✗ Failed to update number:', response.status);
                updateStatusDisplay('Update failed');
                return false;
            }
        } catch (e) {
            console.log('[Lead Source Extension] ✗ Error updating number:', e);
            updateStatusDisplay('Update error');
            return false;
        }
    }

    // Version update UI
    let currentExtensionVersion = null;
    let updateInfo = null;

    function showUpdateWarning(newVersion, downloadUrl) {
        // Store the update info for re-applying if DOM changes
        updateInfo = { newVersion, downloadUrl };

        // Remove existing warning if any
        const existing = document.getElementById('custom-update-warning');
        if (existing) existing.remove();

        const searchBoxContainer = document.querySelector('.TopNav_searchBoxContainer_9a8');
        if (!searchBoxContainer) {
            // Retry later if container not found
            setTimeout(() => showUpdateWarning(newVersion, downloadUrl), 1000);
            return;
        }

        const warning = document.createElement('div');
        warning.id = 'custom-update-warning';
        warning.style.position = 'absolute';
        warning.style.top = '10px';
        warning.style.right = '200px';
        warning.style.display = 'flex';
        warning.style.alignItems = 'center';
        warning.style.gap = '8px';
        warning.style.zIndex = '1000';

        const text = document.createElement('span');
        text.textContent = `v${newVersion} available`;
        text.style.fontSize = '12px';
        text.style.color = '#d97706';
        text.style.fontWeight = '600';

        const btn = document.createElement('button');
        btn.textContent = 'Update';
        btn.style.fontSize = '11px';
        btn.style.padding = '2px 8px';
        btn.style.backgroundColor = '#f59e0b';
        btn.style.color = '#fff';
        btn.style.border = 'none';
        btn.style.borderRadius = '4px';
        btn.style.cursor = 'pointer';
        btn.style.fontWeight = '600';
        btn.onclick = () => window.open(downloadUrl, '_blank');

        warning.appendChild(text);
        warning.appendChild(btn);
        searchBoxContainer.appendChild(warning);
    }

    // Listen for messages from bridge script
    window.addEventListener('message', function(event) {
        if (event.data && event.data.source === 'close-extension-bridge') {
            if (event.data.type === 'CALLER_NAME_INIT' || event.data.type === 'CALLER_NAME_UPDATE') {
                currentCallerName = event.data.callerName;
                console.log('[Lead Source Extension] Caller name set to:', currentCallerName);
            }
            if (event.data.type === 'EXTENSION_VERSION') {
                currentExtensionVersion = event.data.version;
                console.log('[Lead Source Extension] Current version:', currentExtensionVersion);
            }
            if (event.data.type === 'UPDATE_AVAILABLE') {
                console.log('[Lead Source Extension] Update available:', event.data.latestVersion);
                showUpdateWarning(event.data.latestVersion, event.data.downloadUrl);
            }
        }
    });

    // Calendar link click tracking - auto-set status to "Terminiert"
    const TERMINIERT_STATUS_ID = 'stat_WBzPjl29J9uRKWOsSkp0BCv7Bu810vMBQwtcy4YSjV5';

    function isCalendarLink(href) {
        if (!href) return false;
        return href.includes('calendly.com') ||
               href.includes('cal.com') ||
               href.includes('termin.rebound.systems');
    }

    async function updateLeadStatusToTerminiert(leadId) {
        console.log('[Lead Source Extension] Updating lead status to Terminiert:', leadId);

        try {
            const response = await originalFetch(`/api/v1/lead/${leadId}/`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    ...getAuthHeaders()
                },
                body: JSON.stringify({
                    status_id: TERMINIERT_STATUS_ID
                })
            });

            if (response.ok) {
                console.log('[Lead Source Extension] ✓ Lead status updated to Terminiert');
            } else {
                console.log('[Lead Source Extension] ✗ Failed to update status:', response.status);
            }
        } catch (e) {
            console.log('[Lead Source Extension] ✗ Error updating status:', e);
        }
    }

    // Listen for clicks on calendar links in dropdown menus
    document.addEventListener('click', function(event) {
        const link = event.target.closest('a');
        if (!link) return;

        const href = link.getAttribute('href');
        if (!isCalendarLink(href)) return;

        // Get current lead ID from URL
        const leadId = getLeadIdFromUrl();
        if (!leadId) {
            console.log('[Lead Source Extension] Calendar link clicked but no lead ID in URL');
            return;
        }

        console.log('[Lead Source Extension] Calendar link clicked:', href);
        updateLeadStatusToTerminiert(leadId);
    }, true);

    // Store CSRF token and auth headers
    let csrfToken = null;
    let authHeaders = {};

    // Helper to get CSRF token from cookie
    function getCsrfTokenFromCookie() {
        const match = document.cookie.match(/_csrf_token=([^;]+)/);
        return match ? match[1] : null;
    }

    // Helper to get current auth headers (with fallback to cookie)
    function getAuthHeaders() {
        const headers = { ...authHeaders };
        if (!headers['x-csrftoken'] && !headers['X-CSRFToken']) {
            const cookieToken = getCsrfTokenFromCookie();
            if (cookieToken) {
                headers['x-csrftoken'] = cookieToken;
            }
        }
        return headers;
    }

    // Hook XMLHttpRequest in case Close.com uses XHR instead of fetch
    const originalXHROpen = XMLHttpRequest.prototype.open;
    const originalXHRSend = XMLHttpRequest.prototype.send;
    const originalXHRSetRequestHeader = XMLHttpRequest.prototype.setRequestHeader;

    XMLHttpRequest.prototype.open = function(method, url, ...args) {
        this._method = method;
        this._url = url;
        this._headers = {};
        return originalXHROpen.apply(this, [method, url, ...args]);
    };

    XMLHttpRequest.prototype.setRequestHeader = function(header, value) {
        if (this._headers) {
            this._headers[header] = value;
            // Capture auth headers
            if (header.toLowerCase() === 'x-csrftoken' || header.toLowerCase() === 'x-csrf-token') {
                csrfToken = value;
            }
            if (header.toLowerCase() === 'authorization' || header.toLowerCase().startsWith('x-')) {
                authHeaders[header] = value;
            }
        }
        return originalXHRSetRequestHeader.apply(this, arguments);
    };

    XMLHttpRequest.prototype.send = function(body) {
        if (this._method === 'POST' && this._url && this._url.match(/\/api\/v1\/activity\/call\/?$/)) {
            try {
                const bodyData = JSON.parse(body);
                if (bodyData && bodyData.lead_id) {
                    const leadId = bodyData.lead_id;
                    console.log('[Lead Source Extension] ✓ Outbound call detected for lead:', leadId);
                    updateStatusDisplay('Calling...', true);

                    // Handle phone number validation and update
                    const project = projectCache[leadId];
                    if (project) {
                        const requiredNumber = getRequiredPhoneNumber(project);
                        const matchingNumber = findMatchingNumber(requiredNumber);

                        // Check if current number matches required (using normalized comparison)
                        const currentMatchesRequired = currentOutgoingNumber &&
                            normalizePhone(currentOutgoingNumber) === normalizePhone(requiredNumber);

                        // Check if we know the current outgoing number
                        if (currentOutgoingNumber === null) {
                            console.log('[Lead Source Extension] Current number unknown, fetching...');
                            // Hold this request and fetch telephony settings
                            const xhr = this;
                            const originalArgs = arguments;

                            fetchTelephonySettings().then(() => {
                                // Now check if number needs updating (normalized comparison)
                                const currentMatchesAfterFetch = currentOutgoingNumber &&
                                    normalizePhone(currentOutgoingNumber) === normalizePhone(requiredNumber);
                                if (!currentMatchesAfterFetch) {
                                    console.log('[Lead Source Extension] Number mismatch, updating...');
                                    updateOutgoingNumber(requiredNumber).then((success) => {
                                        if (success) {
                                            setTimeout(() => {
                                                updateLastCaller(leadId, currentCallerName, xhr._headers);
                                                originalXHRSend.apply(xhr, originalArgs);
                                            }, 1000);
                                        } else {
                                            // Continue anyway
                                            updateLastCaller(leadId, currentCallerName, xhr._headers);
                                            originalXHRSend.apply(xhr, originalArgs);
                                        }
                                    });
                                } else {
                                    updateLastCaller(leadId, currentCallerName, xhr._headers);
                                    originalXHRSend.apply(xhr, originalArgs);
                                }
                            });
                            return;
                        } else if (!currentMatchesRequired) {
                            // Number needs updating
                            const targetNumber = matchingNumber || requiredNumber;
                            console.log('[Lead Source Extension] Number needs updating from', availableNumbers[currentOutgoingNumber]?.formatted, 'to', availableNumbers[targetNumber]?.formatted || requiredNumber);
                            const xhr = this;
                            const originalArgs = arguments;

                            updateOutgoingNumber(requiredNumber).then((success) => {
                                setTimeout(() => {
                                    updateLastCaller(leadId, currentCallerName, xhr._headers);
                                    originalXHRSend.apply(xhr, originalArgs);
                                }, 1000);
                            });
                            return;
                        } else {
                            console.log('[Lead Source Extension] ✓ Correct number already selected:', availableNumbers[currentOutgoingNumber]?.formatted);
                        }
                    } else {
                        console.log('[Lead Source Extension] Project unknown for lead:', leadId);
                    }

                    updateLastCaller(leadId, currentCallerName, this._headers);
                }
            } catch (e) {
                console.log('[Lead Source Extension] Error parsing call body:', e);
            }
        }
        return originalXHRSend.apply(this, arguments);
    };

    // Extract lead ID from URL
    function getLeadIdFromUrl() {
        const match = location.href.match(/\/lead\/(lead_[^\/]+)/);
        return match ? match[1] : null;
    }

    // Fetch project and last caller directly from API
    async function fetchLeadData(leadId) {
        if (pendingLeadFetches[leadId]) {
            return pendingLeadFetches[leadId];
        }

        const fetchPromise = originalFetch(`/api/v1/lead/${leadId}/`, {
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            const project = data.custom && data.custom[PROJECT_FIELD_NAME];
            const lastCaller = data.custom && data.custom[LAST_CALLER_FIELD_NAME];

            if (project) {
                projectCache[leadId] = project;
            }
            if (lastCaller) {
                lastCallerCache[leadId] = lastCaller;
            }

            // Apply to call bar immediately
            applyToCallBar();

            return { project, lastCaller };
        })
        .catch(e => {
            console.log('[Lead Source Extension] Error fetching lead:', e);
            return null;
        })
        .finally(() => {
            delete pendingLeadFetches[leadId];
        });

        pendingLeadFetches[leadId] = fetchPromise;
        return fetchPromise;
    }

    // Function to update lead's last caller custom field
    async function updateLastCaller(leadId, callerName, headers = {}) {
        const body = {
            [`custom.${LAST_CALLER_FIELD_ID}`]: callerName
        };

        const requestHeaders = {
            'Content-Type': 'application/json',
            ...headers
        };

        try {
            const response = await originalFetch(`/api/v1/lead/${leadId}/`, {
                method: 'PUT',
                headers: requestHeaders,
                body: JSON.stringify(body)
            });

            if (response.ok) {
                console.log('[Lead Source Extension] ✓ Updated last caller to', callerName);

                // Update cache and current state
                lastCallerCache[leadId] = callerName;
                if (leadId === currentLeadId) {
                    currentLastCaller = callerName;
                    // Immediately update the badge
                    addLastCallerBadge(callerName);
                }
            } else {
                console.log('[Lead Source Extension] Failed to update last caller:', response.status);
            }
        } catch (error) {
            console.log('[Lead Source Extension] Error updating last caller:', error);
        }
    }

    // Hook fetch to intercept lead API calls and outbound calls
    const originalFetch = window.fetch;
    window.fetch = function(...args) {
        const request = args[0];
        let url;
        let method;
        let body;
        let fullOptions = args[1];

        if (typeof request === 'string') {
            url = request;
            method = args[1]?.method || 'GET';
            body = args[1]?.body;
        } else if (request instanceof URL) {
            url = request.href;
            method = args[1]?.method || 'GET';
            body = args[1]?.body;
        } else if (request instanceof Request) {
            url = request.url;
            method = request.method;
            body = request.body;
        } else if (request && request.url) {
            url = request.url;
            method = request.method || 'GET';
            body = request.body;
        }

        // Intercept outbound calls via fetch
        if (method === 'POST' && url && url.match(/\/api\/v1\/activity\/call\/?$/)) {
            console.log('[Lead Source Extension] ✓ Call activity POST detected via fetch');
            try {
                const bodyStr = typeof body === 'string' ? body : null;
                if (bodyStr) {
                    const bodyData = JSON.parse(bodyStr);
                    if (bodyData && bodyData.lead_id) {
                        const leadId = bodyData.lead_id;
                        console.log('[Lead Source Extension] ✓ Outbound call detected for lead:', leadId);
                        updateStatusDisplay('Calling...', true);

                        // Handle phone number validation and update
                        const project = projectCache[leadId];
                        if (project) {
                            const requiredNumber = getRequiredPhoneNumber(project);

                            // Check if current number matches required (using normalized comparison)
                            const currentMatchesRequired = currentOutgoingNumber &&
                                normalizePhone(currentOutgoingNumber) === normalizePhone(requiredNumber);

                            if (currentOutgoingNumber === null) {
                                console.log('[Lead Source Extension] Current number unknown, fetching...');
                                // Hold this request and fetch telephony settings first
                                return fetchTelephonySettings().then(() => {
                                    const currentMatchesAfterFetch = currentOutgoingNumber &&
                                        normalizePhone(currentOutgoingNumber) === normalizePhone(requiredNumber);
                                    if (!currentMatchesAfterFetch) {
                                        console.log('[Lead Source Extension] Number mismatch, updating...');
                                        return updateOutgoingNumber(requiredNumber).then((success) => {
                                            return new Promise(resolve => {
                                                setTimeout(() => {
                                                    updateLastCaller(leadId, currentCallerName, getAuthHeaders());
                                                    resolve(originalFetch.apply(this, args));
                                                }, 1000);
                                            });
                                        });
                                    } else {
                                        updateLastCaller(leadId, currentCallerName, getAuthHeaders());
                                        return originalFetch.apply(this, args);
                                    }
                                });
                            } else if (!currentMatchesRequired) {
                                // Number needs updating
                                console.log('[Lead Source Extension] Number needs updating from',
                                    availableNumbers[currentOutgoingNumber]?.formatted, 'to', requiredNumber);
                                return updateOutgoingNumber(requiredNumber).then((success) => {
                                    return new Promise(resolve => {
                                        setTimeout(() => {
                                            updateLastCaller(leadId, currentCallerName, getAuthHeaders());
                                            resolve(originalFetch.apply(this, args));
                                        }, 1000);
                                    });
                                });
                            } else {
                                console.log('[Lead Source Extension] ✓ Correct number already selected:',
                                    availableNumbers[currentOutgoingNumber]?.formatted);
                            }
                        } else {
                            console.log('[Lead Source Extension] Project unknown for lead:', leadId);
                        }

                        updateLastCaller(leadId, currentCallerName, getAuthHeaders());
                    }
                }
            } catch (e) {
                console.log('[Lead Source Extension] Error parsing fetch call body:', e);
            }
        }

        // Intercept GraphQL TelephonySettingsQuery to capture current outgoing number
        if (url && url.includes('/api/v1/graphql/') && url.includes('TelephonySettingsQuery')) {
            return originalFetch.apply(this, args).then(response => {
                const clonedResponse = response.clone();
                clonedResponse.json().then(data => {
                    console.log('[Lead Source Extension] ✓ TelephonySettingsQuery intercepted');
                    processTelephonySettings(data);
                }).catch(e => console.log('[Lead Source Extension] Error processing telephony settings:', e));
                return response;
            });
        }

        // Intercept lead GET requests for project
        if (url && url.includes('/api/v1/lead/lead_')) {
            // Extract lead ID from API URL
            const leadIdMatch = url.match(/\/lead\/(lead_[^\/\?]+)/);
            const apiLeadId = leadIdMatch ? leadIdMatch[1] : null;

            return originalFetch.apply(this, args).then(response => {
                const clonedResponse = response.clone();
                clonedResponse.json().then(data => {
                    const project = data.custom && data.custom[PROJECT_FIELD_NAME];
                    const lastCaller = data.custom && data.custom[LAST_CALLER_FIELD_NAME];

                    if (apiLeadId) {
                        // Cache the project and last caller
                        if (project) {
                            projectCache[apiLeadId] = project;
                        }
                        if (lastCaller) {
                            lastCallerCache[apiLeadId] = lastCaller;
                        }

                        // Only update current if this is for the current URL's lead
                        const urlLeadId = getLeadIdFromUrl();
                        if (apiLeadId === urlLeadId) {
                            currentProject = project;
                            currentLastCaller = lastCaller;
                            currentLeadId = apiLeadId;
                            scheduleApply();
                        }

                        // Try to apply to call bar if visible
                        applyToCallBar();
                    }
                }).catch(e => console.log('[Lead Source Extension] Error:', e));

                return response;
            });
        }

        return originalFetch.apply(this, args);
    };

    function scheduleApply() {
        // Clear any pending application
        if (applyTimeout) {
            clearTimeout(applyTimeout);
        }

        // Apply immediately
        applyCustomizations();

        // Then retry a few times in case DOM isn't ready
        applyTimeout = setTimeout(() => applyCustomizations(), 100);
        setTimeout(() => applyCustomizations(), 300);
        setTimeout(() => applyCustomizations(), 600);
        setTimeout(() => applyCustomizations(), 1000);
    }

    function applyBackgroundColor(project) {
        const color = projectColors[project] || projectColors['Default'];
        const appContent = document.querySelector('.app-content');

        if (appContent) {
            if (appContent.style.backgroundColor !== color) {
                appContent.style.backgroundColor = color;
                appContent.style.transition = 'background-color 0.3s ease';
            }
            return true;
        }
        return false;
    }

    function addProjectBadge(project) {
        if (!project) return false;

        // Check if badge already exists and is correct
        const existingBadge = document.getElementById('custom-project-badge');
        if (existingBadge && existingBadge.textContent === project) {
            return true;
        }

        if (existingBadge) {
            existingBadge.remove();
        }

        // Find the h1 element (use pattern matching for class names that change)
        const nameRowContainer = document.querySelector('[class*="LeadHeader_mainContent"] [class*="structure_utils_root"] h1');

        if (nameRowContainer) {
            const color = projectColors[project] || projectColors['Default'];

            const badge = document.createElement('span');
            badge.id = 'custom-project-badge';
            badge.textContent = project;
            badge.style.fontSize = '14px';
            badge.style.fontWeight = '600';
            badge.style.color = '#333';
            badge.style.backgroundColor = color;
            badge.style.padding = '4px 10px';
            badge.style.marginLeft = '12px';
            badge.style.borderRadius = '4px';
            badge.style.display = 'inline-block';
            badge.style.verticalAlign = 'middle';
            badge.style.border = '1px solid rgba(0, 0, 0, 0.1)';

            // Insert right after the h1 element
            nameRowContainer.insertAdjacentElement('afterend', badge);
            return true;
        }
        return false;
    }

    function addLastCallerBadge(lastCaller) {
        if (!lastCaller) return false;

        // Check if badge already exists and is correct
        const existingBadge = document.getElementById('custom-last-caller-badge');
        if (existingBadge && existingBadge.textContent === lastCaller) {
            return true;
        }

        if (existingBadge) {
            existingBadge.remove();
        }

        // Find the project badge to insert after it, or fall back to h1
        const projectBadge = document.getElementById('custom-project-badge');
        const insertAfter = projectBadge || document.querySelector('[class*="LeadHeader_mainContent"] [class*="structure_utils_root"] h1');

        if (insertAfter) {
            const badge = document.createElement('span');
            badge.id = 'custom-last-caller-badge';
            badge.textContent = lastCaller;
            badge.style.fontSize = '14px';
            badge.style.fontWeight = '600';
            badge.style.color = '#333';
            badge.style.backgroundColor = '#E8EAF6';  // Light indigo
            badge.style.padding = '4px 10px';
            badge.style.marginLeft = '8px';
            badge.style.borderRadius = '4px';
            badge.style.display = 'inline-block';
            badge.style.verticalAlign = 'middle';
            badge.style.border = '1px solid rgba(0, 0, 0, 0.1)';

            // Insert right after the lead source badge (or h1 if lead source badge doesn't exist)
            insertAfter.insertAdjacentElement('afterend', badge);
            return true;
        }
        return false;
    }

    function applyToCallBar() {
        // Call bar feature disabled - Close.com doesn't use links in the call bar
        return false;

        const leadIdMatch = leadLink.href.match(/\/lead\/(lead_[^\/]+)/);
        if (!leadIdMatch) {
            return false;
        }

        const leadId = leadIdMatch[1];
        let project = projectCache[leadId];

        if (!project) {
            // Fetch the project directly (only if not already fetching)
            if (!pendingLeadFetches[leadId]) {
                fetchLeadData(leadId);
            }
            return false;
        }

        // Check if badge already exists
        const existingBadge = callBar.querySelector('#custom-call-bar-project-badge');
        if (existingBadge && existingBadge.textContent === project) {
            return true;
        }

        if (existingBadge) {
            existingBadge.remove();
        }

        // Find the lead row container
        const leadRow = callBar.querySelector('.components_Contact_leadRow_d10');
        if (!leadRow) {
            return false;
        }

        const color = projectColors[project] || projectColors['Default'];

        const badge = document.createElement('span');
        badge.id = 'custom-call-bar-project-badge';
        badge.textContent = project;
        badge.className = 'Badge_outer_472 Badge_badge_7e7 Badge_small_877 Badge_root_140';
        badge.style.backgroundColor = color;
        badge.style.color = '#333';
        badge.style.fontSize = '12px';
        badge.style.fontWeight = '600';
        badge.style.padding = '2px 8px';
        badge.style.marginLeft = '8px';
        badge.style.borderRadius = '4px';
        badge.style.display = 'inline-block';
        badge.style.border = '1px solid rgba(0, 0, 0, 0.1)';

        leadRow.appendChild(badge);
        return true;
    }

    function applyCustomizations() {
        // Prevent concurrent applications
        if (isApplying) return;
        isApplying = true;

        if (!currentProject) {
            isApplying = false;
            return;
        }

        // Verify we're still on the same lead
        const urlLeadId = getLeadIdFromUrl();
        if (urlLeadId !== currentLeadId) {
            isApplying = false;
            return;
        }

        const bgApplied = applyBackgroundColor(currentProject);
        const badgeApplied = addProjectBadge(currentProject);
        const lastCallerApplied = addLastCallerBadge(currentLastCaller);

        isApplying = false;
    }

    // Add overlay to phone settings popover to block phone number selection
    function addPhoneSettingsOverlay() {
        // Find the Phone Settings popover
        const popovers = document.querySelectorAll('[data-popover="true"][class*="Popover_popover"]');

        for (const popover of popovers) {
            // Check if this is the Phone Settings popover
            const header = popover.querySelector('[class*="SupportedPlan_header"]');
            if (!header) continue;

            const headerText = header.textContent;
            if (!headerText || !headerText.includes('Phone Settings')) continue;

            // Find the Caller ID section
            const callerIdLabel = popover.querySelector('span.typography_boldWeight_f2a');
            if (!callerIdLabel || !callerIdLabel.textContent.includes('Caller ID')) continue;

            // Find the select input container (the one with the phone number)
            const selectContainer = popover.querySelector('[class*="v2_container"]');
            if (!selectContainer) continue;

            // Check if overlay already exists
            if (selectContainer.querySelector('#custom-phone-select-overlay')) continue;

            // Make container position relative for overlay positioning
            selectContainer.style.position = 'relative';

            // Create the overlay
            const overlay = document.createElement('div');
            overlay.id = 'custom-phone-select-overlay';
            overlay.style.position = 'absolute';
            overlay.style.top = '0';
            overlay.style.left = '0';
            overlay.style.right = '0';
            overlay.style.bottom = '0';
            overlay.style.backgroundColor = 'rgba(255, 255, 255, 0.9)';
            overlay.style.display = 'flex';
            overlay.style.alignItems = 'center';
            overlay.style.justifyContent = 'center';
            overlay.style.zIndex = '100';
            overlay.style.borderRadius = '4px';
            overlay.style.cursor = 'not-allowed';

            const message = document.createElement('span');
            message.textContent = 'Outbound Nummer wird über Extension gesteuert';
            message.style.fontSize = '12px';
            message.style.fontWeight = '600';
            message.style.color = '#666';
            message.style.textAlign = 'center';
            message.style.padding = '4px 8px';

            overlay.appendChild(message);
            selectContainer.appendChild(overlay);

            console.log('[Lead Source Extension] ✓ Phone settings overlay added');
        }
    }

    // Watch for DOM changes to reapply badge if needed
    const observer = new MutationObserver((mutations) => {
        const nameContainer = document.querySelector('[class*="LeadHeader_mainContent"] [class*="structure_utils_root"] h1');
        if (nameContainer) {
            if (currentProject && !document.getElementById('custom-project-badge')) {
                applyCustomizations();
            }
            if (currentLastCaller && !document.getElementById('custom-last-caller-badge')) {
                applyCustomizations();
            }
        }

        // Check for TopNav and create status display if needed
        const searchBoxContainer = document.querySelector('.TopNav_searchBoxContainer_9a8');
        if (searchBoxContainer && !document.getElementById('custom-phone-status')) {
            if (currentOutgoingNumber && availableNumbers[currentOutgoingNumber]) {
                createStatusDisplay(availableNumbers[currentOutgoingNumber].formatted);
            } else {
                createStatusDisplay('Loading...');
            }
        }

        // Re-apply update warning if it was removed
        if (searchBoxContainer && updateInfo && !document.getElementById('custom-update-warning')) {
            showUpdateWarning(updateInfo.newVersion, updateInfo.downloadUrl);
        }

        // Check for phone settings popover and add overlay
        addPhoneSettingsOverlay();

        // Check for call bar and apply badge if needed
        applyToCallBar();
    });

    // Start observing once DOM is ready
    if (document.body) {
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
        // Try to create status display immediately if TopNav exists
        setTimeout(() => {
            const searchBoxContainer = document.querySelector('.TopNav_searchBoxContainer_9a8');
            if (searchBoxContainer && !document.getElementById('custom-phone-status')) {
                createStatusDisplay('Loading...');
            }
        }, 1000);
    } else {
        document.addEventListener('DOMContentLoaded', () => {
            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
            // Try to create status display after DOM loads
            setTimeout(() => {
                const searchBoxContainer = document.querySelector('.TopNav_searchBoxContainer_9a8');
                if (searchBoxContainer && !document.getElementById('custom-phone-status')) {
                    createStatusDisplay('Loading...');
                }
            }, 1000);
        });
    }

    // Monitor URL changes
    let lastUrl = location.href;
    setInterval(() => {
        if (location.href !== lastUrl) {
            lastUrl = location.href;

            // Clear old state
            if (applyTimeout) clearTimeout(applyTimeout);

            // Remove old customizations
            const projectBadge = document.getElementById('custom-project-badge');
            if (projectBadge) projectBadge.remove();

            const lastCallerBadge = document.getElementById('custom-last-caller-badge');
            if (lastCallerBadge) lastCallerBadge.remove();

            const appContent = document.querySelector('.app-content');
            if (appContent) appContent.style.backgroundColor = '';

            // Reset state - API call will set new values
            currentProject = null;
            currentLastCaller = null;
            currentLeadId = null;
        }

        // Always try to apply to call bar if it's visible
        applyToCallBar();
    }, 200);

    console.log('[Lead Source Extension] Ready');
})();
