// Background service worker for update checking (bypasses CORS)

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'CHECK_FOR_UPDATES') {
    checkForUpdates(message.currentVersion).then(sendResponse);
    return true; // Keep channel open for async response
  }
});

function compareVersions(v1, v2) {
  const parts1 = v1.split('.').map(Number);
  const parts2 = v2.split('.').map(Number);
  const maxLen = Math.max(parts1.length, parts2.length);

  for (let i = 0; i < maxLen; i++) {
    const p1 = parts1[i] || 0;
    const p2 = parts2[i] || 0;
    if (p1 > p2) return 1;
    if (p1 < p2) return -1;
  }
  return 0;
}

async function checkForUpdates(currentVersion) {
  try {
    const response = await fetch('https://intern.rebound.systems/extension/');
    const html = await response.text();

    // Parse zip filenames from the index page
    const zipRegex = /<a href="([^"]+\.zip)">/g;
    let match;
    let versions = [];

    while ((match = zipRegex.exec(html)) !== null) {
      const filename = match[1];
      const version = filename.replace('.zip', '');
      versions.push({ version, filename });
    }

    // Find the highest version
    versions.sort((a, b) => compareVersions(b.version, a.version));

    if (versions.length > 0) {
      const newest = versions[0];
      const latestVersion = newest.version;
      const latestVersionUrl = `https://intern.rebound.systems/extension/${newest.filename}`;

      if (compareVersions(latestVersion, currentVersion) > 0) {
        console.log('[Close Extension] Update available:', latestVersion, '(current:', currentVersion + ')');
        return {
          updateAvailable: true,
          latestVersion,
          downloadUrl: latestVersionUrl
        };
      } else {
        console.log('[Close Extension] Up to date:', currentVersion);
        return { updateAvailable: false };
      }
    }
    return { updateAvailable: false };
  } catch (e) {
    console.log('[Close Extension] Error checking for updates:', e);
    return { updateAvailable: false, error: e.message };
  }
}
